<?php
/* --------------------------------------------------------------
 MenuFactory.php 2020-04-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 30 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Layout\Menu\Factories;

use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Admin\Layout\Menu\Filter\Condition;
use Gambio\Admin\Layout\Menu\Models\Cache\Collections\MenuItems;
use Gambio\Admin\Layout\Menu\Models\Cache\MenuGroup;
use Gambio\Admin\Layout\Menu\Models\Cache\MenuItem;
use Gambio\Core\Language\TextManager;
use InvalidArgumentException;
use Webmozart\Assert\Assert;
use function array_key_exists;

/**
 * Class MenuFactory
 * @package Gambio\Admin\Layout\Menu
 *
 *          I decided to skip unit tests for this class, because it should be safe to use.
 *          I mean by "safe to use", that the methods either return a freshly created models, or fail
 *          due to invalid data provided. But there is not much space to perform useful tests and the
 *          type/exception system ensure correct usage.
 *
 * @codeCoverageIgnore
 */
class CacheMenuFactory
{
    /**
     * @var TextManager
     */
    private $textManager;
    
    /**
     * @var int
     */
    private $languageId;
    
    
    /**
     * MenuFactory constructor.
     *
     * @param TextManager     $textManager
     * @param UserPreferences $userPreferences
     */
    public function __construct(TextManager $textManager, UserPreferences $userPreferences)
    {
        $this->textManager = $textManager;
        $this->languageId  = $userPreferences->languageId();
    }
    
    
    /**
     * Creates a condition if possible.
     *
     * This function ties to create a new condition model from the given data.
     * If anything goes wrong, just null will be returned.
     * This can be the case too, if the "if" key not exists in the data set, which means that no
     * filter should be applied.
     *
     * @param array $data
     *
     * @return Condition|null
     */
    public function createCondition(array $data): ?Condition
    {
        if (!array_key_exists('if', $data)) {
            return null;
        }
        
        try {
            $condition = $data['if'];
            Assert::keyExists($condition, 'filter');
            Assert::keyExists($condition, 'args');
            
            Assert::string($condition['filter']);
            Assert::isArray($condition['args']);
            
            return Condition::create($condition['filter'], $condition['args']);
        } catch (InvalidArgumentException $e) {
            return null;
        }
    }
    
    
    /**
     * Creates a menu group.
     *
     * This function creates a menu group model from the given data.
     * In case of bad formed $data, an exception is thrown.
     *
     * @param array          $data
     * @param Condition|null $condition
     *
     * @return MenuGroup
     * @throws InvalidArgumentException
     */
    public function createMenuGroup(array $data, ?Condition $condition): MenuGroup
    {
        $data['title'] = $this->translate($data['title']);
        
        return MenuGroup::fromArray($data, new MenuItems(), $condition);
    }
    
    
    /**
     * Creates a menu item.
     *
     * This function creates a menu item from the given data.
     * In case of bad formed $data, an exception is thrown.
     *
     * @param array          $data
     * @param Condition|null $condition
     *
     * @return MenuItem
     * @throws InvalidArgumentException
     */
    public function createMenuItem(array $data, ?Condition $condition): MenuItem
    {
        $data['title'] = $this->translate($data['title']);
        
        return MenuItem::fromArray($data, $condition);
    }
    
    
    /**
     * Translates the language key.
     *
     * @param string $langKey
     *
     * @return string
     */
    private function translate(string $langKey): string
    {
        $langData = explode('.', $langKey);
        if (count($langData) === 2) {
            [$section, $phrase] = $langData;
            
            $translated = $this->textManager->getPhraseText($phrase, $section, $this->languageId);
            
            return $translated !== $phrase ? $translated : $langKey;
        }
        
        return $langKey;
    }
}